function [J,E,S,n_iter] = NR_stott(S_star,E_star,Y,E_0,idx,Parameters)
%
% INPUT
% - Y           nodal admittance matrix
% - S_star      given complex powers (active/reactive powers)
% - E_star      given voltage magnitudes
% - E_0         initial voltages (phasors)
% - idx.slack   index of the slack bus
% - idx.pq      indices of the PQ buses
% - idx.pv      indices of the PV buses
% - Parameters.tol         tolerance for convergence criterion
% - Parameters.n_max       maximum number of iterations
%
% OUTPUT
% - E           solution voltages (phasors)
% - J           Jacobian at the solution
% - n_iter      number of iterations

G = real(Y);
B = imag(Y);

% Initialization
Eabs = abs(E_0);
Eabs(idx.pv) = E_star(idx.pv); % Set given values for PV nodes
Earg = angle(E_0);

J = [];

% Defining the submatrics B' and B'' 
% B'
B_d1 = -B;
% B'' 
B_d2 = -B;

% removing slack node 

B_d1(:,idx.slack) = [];
B_d1(idx.slack,:) = [];

B_d2(:,idx.slack) = [];
B_d2(idx.slack,:) = [];

% constructing Jacobian

J = [zeros(size(B_d1)) B_d1
    B_d2 zeros(size(B_d2))];

for k=1:Parameters.n_max
    n_iter = k;
    
    % Compute nodal voltages/currents/power
    E = complex(Eabs.*cos(Earg),Eabs.*sin(Earg));
    I = Y*E;
    S = E.*conj(I);
    
    %% Mismatch calculation
    
    % Compute the mismatches for the entire network.
    dS = S_star-S;
    dP = real(dS);
    dQ = imag(dS);
    
    % Keep only the relevant mismatches.
    dP(idx.slack) = [];
    dQ(sort([idx.pv;idx.slack])) = [];
    
    dF = [dP./Eabs(idx.pq);dQ./Eabs(idx.pq)]; % mismatch of the power flow equations
    
    %% Convergence check
    
    if(max(abs(dF))<Parameters.tol)
        disp('NR algorithm has converged to a solution!');
        break;
    elseif(k==Parameters.n_max)
        disp('NR algorithm reached the maximum number of iterations!');
    end

    
    %% Solution update
    
    % Solve
    dx = J \ dF;
    
    % Reconstruct the solution
    
    dEabs = zeros(length(Eabs),1);
    dEabs(idx.pq,1) = dx(1:length(idx.pq));
    
    dEarg = zeros(length(Earg),1);
    dEarg(sort([idx.pq;idx.pv]),1) = dx((length(idx.pq)+1):end);
    
    % Update
    Eabs = Eabs + dEabs;
    Earg = Earg + dEarg;
end

E = Eabs .* exp(1i*Earg);

end